/*
 * Decompiled with CFR 0.152.
 */
package net.sf.picard.illumina;

import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import net.sf.picard.PicardException;
import net.sf.picard.cmdline.CommandLineProgram;
import net.sf.picard.cmdline.Option;
import net.sf.picard.cmdline.Usage;
import net.sf.picard.illumina.parser.IlluminaDataProvider;
import net.sf.picard.illumina.parser.IlluminaDataProviderFactory;
import net.sf.picard.illumina.parser.IlluminaDataType;
import net.sf.picard.illumina.parser.IlluminaReadData;
import net.sf.picard.io.IoUtil;
import net.sf.picard.metrics.MetricBase;
import net.sf.picard.metrics.MetricsFile;
import net.sf.picard.util.TabbedTextFileWithHeaderParser;
import net.sf.samtools.util.SequenceUtil;
import net.sf.samtools.util.StringUtil;

public class ExtractIlluminaBarcodes
extends CommandLineProgram {
    @Usage
    public String USAGE = this.getStandardUsagePreamble() + "Determine the barcode for each read in an Illumina lane.\n" + "For each tile, a file is written to the basecalls directory of the form s_<lane>_<tile>_barcode.txt." + "An output file contains a line for each read in the tile, aligned with the regular basecall output\n" + "The output file contains the following tab-separated columns: \n" + "    * read subsequence at barcode position\n" + "    * Y or N indicating if there was a barcode match\n" + "    * matched barcode sequence\n" + "Note that the order of specification of barcodes can cause arbitrary differences in output for poorly matching barcodes.\n\n";
    @Option(doc="Deprecated option; use BASECALLS_DIR", mutex={"BASECALLS_DIR"})
    public File BUSTARD_DIR;
    @Option(doc="The Illumina basecalls output directory. ", mutex={"BUSTARD_DIR"}, shortName="B")
    public File BASECALLS_DIR;
    @Option(doc="Where to write _barcode.txt files.  By default, these are written to BASECALLS_DIR.", optional=true)
    public File OUTPUT_DIR;
    @Option(doc="Lane number. ", shortName="L")
    public Integer LANE;
    @Option(doc="1-based cycle number of the start of the barcode.", shortName="BARCODE_POSITION")
    public Integer BARCODE_CYCLE;
    @Option(doc="Barcode sequence.  These must be unique, and all the same length.", mutex={"BARCODE_FILE"})
    public List<String> BARCODE = new ArrayList<String>();
    @Option(doc="Tab-delimited file of barcode sequences, and optionally barcode name and library name.  Barcodes must be unique, and all the same length.  Column headers must be 'barcode_sequence', 'barcode_name', and 'library_name'.", mutex={"BARCODE"})
    public File BARCODE_FILE;
    @Option(doc="Per-barcode and per-lane metrics written to this file.", shortName="M")
    public File METRICS_FILE;
    @Option(doc="Maximum mismatches for a barcode to be considered a match.")
    public int MAX_MISMATCHES = 1;
    @Option(doc="Minimum difference between number of mismatches in the best and second best barcodes for a barcode to be considered a match.")
    public int MIN_MISMATCH_DELTA = 1;
    @Option(doc="Maximum allowable number of no-calls in a barcode read before it is considered unmatchable.")
    public int MAX_NO_CALLS = 2;
    private int barcodeLength;
    private int tile = 0;
    private File barcodeFile = null;
    private BufferedWriter writer = null;
    private final List<NamedBarcode> namedBarcodes = new ArrayList<NamedBarcode>();
    private final List<BarcodeMetric> barcodeMetrics = new ArrayList<BarcodeMetric>();
    private BarcodeMetric noMatchBarcodeMetric;
    private final NumberFormat tileNumberFormatter = NumberFormat.getNumberInstance();
    private static final String BARCODE_SEQUENCE_COLUMN = "barcode_sequence";
    private static final String BARCODE_NAME_COLUMN = "barcode_name";
    private static final String LIBRARY_NAME_COLUMN = "library_name";

    public ExtractIlluminaBarcodes() {
        this.tileNumberFormatter.setMinimumIntegerDigits(4);
        this.tileNumberFormatter.setGroupingUsed(false);
    }

    @Override
    protected int doWork() {
        if (this.BUSTARD_DIR != null) {
            this.BASECALLS_DIR = this.BUSTARD_DIR;
        }
        IoUtil.assertDirectoryIsWritable(this.BASECALLS_DIR);
        IoUtil.assertFileIsWritable(this.METRICS_FILE);
        if (this.OUTPUT_DIR == null) {
            this.OUTPUT_DIR = this.BASECALLS_DIR;
        }
        IoUtil.assertDirectoryIsWritable(this.OUTPUT_DIR);
        for (NamedBarcode namedBarcode : this.namedBarcodes) {
            this.barcodeMetrics.add(new BarcodeMetric(namedBarcode));
        }
        StringBuilder noMatchBarcode = new StringBuilder(this.barcodeLength);
        for (int i = 0; i < this.barcodeLength; ++i) {
            noMatchBarcode.append('N');
        }
        this.noMatchBarcodeMetric = new BarcodeMetric(new NamedBarcode(noMatchBarcode.toString()));
        IlluminaDataProviderFactory factory = new IlluminaDataProviderFactory(this.BASECALLS_DIR, (int)this.LANE, this.BARCODE_CYCLE, this.barcodeLength, IlluminaDataType.BaseCalls, IlluminaDataType.PF);
        factory.setAllowZeroLengthFirstEnd(true);
        IlluminaDataProvider parser = factory.makeDataProvider();
        try {
            while (parser.hasNext()) {
                IlluminaReadData ird = parser.next();
                this.extractBarcode(ird);
            }
            if (this.writer != null) {
                this.writer.close();
                this.writer = null;
            }
        }
        catch (IOException e) {
            throw new PicardException("IOException writing barcode file " + this.barcodeFile, e);
        }
        int totalReads = this.noMatchBarcodeMetric.READS;
        int totalPfReads = this.noMatchBarcodeMetric.PF_READS;
        int totalPfReadsAssigned = 0;
        for (BarcodeMetric barcodeMetric : this.barcodeMetrics) {
            totalReads += barcodeMetric.READS;
            totalPfReads += barcodeMetric.PF_READS;
            totalPfReadsAssigned += barcodeMetric.PF_READS;
        }
        if (totalReads > 0) {
            this.noMatchBarcodeMetric.PCT_MATCHES = (double)this.noMatchBarcodeMetric.READS / (double)totalReads;
            double bestPctOfAllBarcodeMatches = 0.0;
            for (BarcodeMetric barcodeMetric : this.barcodeMetrics) {
                barcodeMetric.PCT_MATCHES = (double)barcodeMetric.READS / (double)totalReads;
                if (!(barcodeMetric.PCT_MATCHES > bestPctOfAllBarcodeMatches)) continue;
                bestPctOfAllBarcodeMatches = barcodeMetric.PCT_MATCHES;
            }
            if (bestPctOfAllBarcodeMatches > 0.0) {
                this.noMatchBarcodeMetric.RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = this.noMatchBarcodeMetric.PCT_MATCHES / bestPctOfAllBarcodeMatches;
                for (BarcodeMetric barcodeMetric : this.barcodeMetrics) {
                    barcodeMetric.RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = barcodeMetric.PCT_MATCHES / bestPctOfAllBarcodeMatches;
                }
            }
        }
        if (totalPfReads > 0) {
            this.noMatchBarcodeMetric.PF_PCT_MATCHES = (double)this.noMatchBarcodeMetric.PF_READS / (double)totalPfReads;
            double bestPctOfAllBarcodeMatches = 0.0;
            for (BarcodeMetric barcodeMetric : this.barcodeMetrics) {
                barcodeMetric.PF_PCT_MATCHES = (double)barcodeMetric.PF_READS / (double)totalPfReads;
                if (!(barcodeMetric.PF_PCT_MATCHES > bestPctOfAllBarcodeMatches)) continue;
                bestPctOfAllBarcodeMatches = barcodeMetric.PF_PCT_MATCHES;
            }
            if (bestPctOfAllBarcodeMatches > 0.0) {
                this.noMatchBarcodeMetric.PF_RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = this.noMatchBarcodeMetric.PF_PCT_MATCHES / bestPctOfAllBarcodeMatches;
                for (BarcodeMetric barcodeMetric : this.barcodeMetrics) {
                    barcodeMetric.PF_RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = barcodeMetric.PF_PCT_MATCHES / bestPctOfAllBarcodeMatches;
                }
            }
        }
        if (totalPfReadsAssigned > 0) {
            double mean = (double)totalPfReadsAssigned / (double)this.barcodeMetrics.size();
            for (BarcodeMetric m : this.barcodeMetrics) {
                m.PF_NORMALIZED_MATCHES = (double)m.PF_READS / mean;
            }
        }
        MetricsFile metrics = this.getMetricsFile();
        for (BarcodeMetric barcodeMetric : this.barcodeMetrics) {
            metrics.addMetric(barcodeMetric);
        }
        metrics.addMetric(this.noMatchBarcodeMetric);
        metrics.write(this.METRICS_FILE);
        return 0;
    }

    private void ensureBarcodeFileOpen(int tile) {
        if (tile == this.tile) {
            return;
        }
        try {
            if (this.writer != null) {
                this.writer.close();
                this.writer = null;
            }
            this.tile = tile;
            this.barcodeFile = this.getBarcodeFile(tile);
            this.writer = IoUtil.openFileForBufferedWriting(this.barcodeFile);
        }
        catch (IOException e) {
            throw new PicardException("IOException " + this.barcodeFile, e);
        }
    }

    private void extractBarcode(IlluminaReadData ird) throws IOException {
        String barcodeSubsequence = StringUtil.bytesToString((byte[])ird.getBarcodeRead().getBases());
        boolean passingFilter = ird.isPf();
        BarcodeMatch match = this.findBestBarcode(barcodeSubsequence, passingFilter);
        String yOrN = match.matched ? "Y" : "N";
        this.ensureBarcodeFileOpen(ird.getTile());
        this.writer.write(StringUtil.join((String)"\t", (String[])new String[]{barcodeSubsequence, yOrN, match.barcode, String.valueOf(match.mismatches), String.valueOf(match.mismatchesToSecondBest)}));
        this.writer.newLine();
    }

    private BarcodeMatch findBestBarcode(String readSubsequence, boolean passingFilter) {
        BarcodeMetric bestBarcodeMetric = null;
        int numMismatchesInBestBarcode = readSubsequence.length();
        int numMismatchesInSecondBestBarcode = readSubsequence.length();
        byte[] readBytes = StringUtil.stringToBytes((String)readSubsequence);
        int numNoCalls = 0;
        for (byte b : readBytes) {
            if (!SequenceUtil.isNoCall((byte)b)) continue;
            ++numNoCalls;
        }
        for (BarcodeMetric barcodeMetric : this.barcodeMetrics) {
            int numMismatches = this.countMismatches(barcodeMetric.barcodeBytes, readBytes);
            if (numMismatches < numMismatchesInBestBarcode) {
                if (bestBarcodeMetric != null) {
                    numMismatchesInSecondBestBarcode = numMismatchesInBestBarcode;
                }
                numMismatchesInBestBarcode = numMismatches;
                bestBarcodeMetric = barcodeMetric;
                continue;
            }
            if (numMismatches >= numMismatchesInSecondBestBarcode) continue;
            numMismatchesInSecondBestBarcode = numMismatches;
        }
        boolean matched = bestBarcodeMetric != null && numNoCalls <= this.MAX_NO_CALLS && numMismatchesInBestBarcode <= this.MAX_MISMATCHES && numMismatchesInSecondBestBarcode - numMismatchesInBestBarcode >= this.MIN_MISMATCH_DELTA;
        BarcodeMatch match = new BarcodeMatch();
        if (numNoCalls + numMismatchesInBestBarcode < readSubsequence.length()) {
            match.mismatches = numMismatchesInBestBarcode;
            match.mismatchesToSecondBest = numMismatchesInSecondBestBarcode;
            match.barcode = bestBarcodeMetric.BARCODE.toLowerCase();
        } else {
            match.mismatches = readSubsequence.length();
            match.mismatches = readSubsequence.length();
            match.barcode = "";
        }
        if (matched) {
            ++bestBarcodeMetric.READS;
            if (passingFilter) {
                ++bestBarcodeMetric.PF_READS;
            }
            if (numMismatchesInBestBarcode == 0) {
                ++bestBarcodeMetric.PERFECT_MATCHES;
                if (passingFilter) {
                    ++bestBarcodeMetric.PF_PERFECT_MATCHES;
                }
            } else if (numMismatchesInBestBarcode == 1) {
                ++bestBarcodeMetric.ONE_MISMATCH_MATCHES;
                if (passingFilter) {
                    ++bestBarcodeMetric.PF_ONE_MISMATCH_MATCHES;
                }
            }
            match.matched = true;
            match.barcode = bestBarcodeMetric.BARCODE;
        } else {
            ++this.noMatchBarcodeMetric.READS;
            if (passingFilter) {
                ++this.noMatchBarcodeMetric.PF_READS;
            }
        }
        return match;
    }

    private int countMismatches(byte[] barcodeBytes, byte[] readSubsequence) {
        int numMismatches = 0;
        for (int i = 0; i < barcodeBytes.length; ++i) {
            if (SequenceUtil.isNoCall((byte)readSubsequence[i]) || SequenceUtil.basesEqual((byte)barcodeBytes[i], (byte)readSubsequence[i])) continue;
            ++numMismatches;
        }
        return numMismatches;
    }

    private File getBarcodeFile(int tile) {
        return new File(this.OUTPUT_DIR, "s_" + this.LANE + "_" + this.tileNumberFormatter.format(tile) + "_barcode.txt");
    }

    @Override
    protected String[] customCommandLineValidation() {
        ArrayList<String> messages = new ArrayList<String>();
        if (this.BARCODE_CYCLE < 1) {
            messages.add("Invalid BARCODE_POSITION=" + this.BARCODE_CYCLE + ".  Value must be positive.");
        }
        if (this.BARCODE_FILE != null) {
            this.parseBarcodeFile(messages);
        } else {
            HashSet<String> barcodes = new HashSet<String>();
            this.barcodeLength = this.BARCODE.get(0).length();
            for (String barcode : this.BARCODE) {
                if (barcode.length() != this.barcodeLength) {
                    messages.add("Barcode " + barcode + " has different length from first barcode.");
                }
                if (barcodes.contains(barcode)) {
                    messages.add("Barcode " + barcode + " specified more than once.");
                }
                barcodes.add(barcode);
                NamedBarcode namedBarcode = new NamedBarcode(barcode);
                this.namedBarcodes.add(namedBarcode);
            }
        }
        if (this.namedBarcodes.size() == 0) {
            messages.add("No barcodes have been specified.");
        }
        if (messages.size() == 0) {
            return null;
        }
        return messages.toArray(new String[messages.size()]);
    }

    public static void main(String[] argv) {
        System.exit(new ExtractIlluminaBarcodes().instanceMain(argv));
    }

    private void parseBarcodeFile(ArrayList<String> messages) {
        TabbedTextFileWithHeaderParser barcodesParser = new TabbedTextFileWithHeaderParser(this.BARCODE_FILE);
        if (!barcodesParser.hasColumn(BARCODE_SEQUENCE_COLUMN)) {
            messages.add(this.BARCODE_FILE + " does not have " + BARCODE_SEQUENCE_COLUMN + " column header");
            return;
        }
        boolean hasBarcodeName = barcodesParser.hasColumn(BARCODE_NAME_COLUMN);
        boolean hasLibraryName = barcodesParser.hasColumn(LIBRARY_NAME_COLUMN);
        this.barcodeLength = 0;
        HashSet<String> barcodes = new HashSet<String>();
        for (TabbedTextFileWithHeaderParser.Row row : barcodesParser) {
            String barcode = row.getField(BARCODE_SEQUENCE_COLUMN);
            if (this.barcodeLength == 0) {
                this.barcodeLength = barcode.length();
            }
            if (barcode.length() != this.barcodeLength) {
                messages.add("Barcode " + barcode + " has different length from first barcode.");
            }
            if (barcodes.contains(barcode)) {
                messages.add("Barcode " + barcode + " specified more than once in " + this.BARCODE_FILE);
            }
            barcodes.add(barcode);
            String barcodeName = hasBarcodeName ? row.getField(BARCODE_NAME_COLUMN) : "";
            String libraryName = hasLibraryName ? row.getField(LIBRARY_NAME_COLUMN) : "";
            NamedBarcode namedBarcode = new NamedBarcode(barcode, barcodeName, libraryName);
            this.namedBarcodes.add(namedBarcode);
        }
    }

    public static class BarcodeMetric
    extends MetricBase {
        public String BARCODE;
        public String BARCODE_NAME = "";
        public String LIBRARY_NAME = "";
        public int READS = 0;
        public int PF_READS = 0;
        public int PERFECT_MATCHES = 0;
        public int PF_PERFECT_MATCHES = 0;
        public int ONE_MISMATCH_MATCHES = 0;
        public int PF_ONE_MISMATCH_MATCHES = 0;
        public double PCT_MATCHES = 0.0;
        public double RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = 0.0;
        public double PF_PCT_MATCHES = 0.0;
        public double PF_RATIO_THIS_BARCODE_TO_BEST_BARCODE_PCT = 0.0;
        public double PF_NORMALIZED_MATCHES;
        protected final byte[] barcodeBytes;

        public BarcodeMetric(NamedBarcode namedBarcode) {
            this.BARCODE = namedBarcode.barcode;
            this.BARCODE_NAME = namedBarcode.barcodeName;
            this.LIBRARY_NAME = namedBarcode.libraryName;
            this.barcodeBytes = StringUtil.stringToBytes((String)this.BARCODE);
        }

        public BarcodeMetric() {
            this.barcodeBytes = null;
        }
    }

    private static class NamedBarcode {
        public final String barcode;
        public final String barcodeName;
        public final String libraryName;

        public NamedBarcode(String barcode, String barcodeName, String libraryName) {
            this.barcode = barcode;
            this.barcodeName = barcodeName;
            this.libraryName = libraryName;
        }

        public NamedBarcode(String barcode) {
            this.barcode = barcode;
            this.barcodeName = "";
            this.libraryName = "";
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            NamedBarcode that = (NamedBarcode)o;
            return !(this.barcode != null ? !this.barcode.equals(that.barcode) : that.barcode != null);
        }

        public int hashCode() {
            return this.barcode != null ? this.barcode.hashCode() : 0;
        }
    }

    static class BarcodeMatch {
        boolean matched;
        String barcode;
        int mismatches;
        int mismatchesToSecondBest;

        BarcodeMatch() {
        }
    }
}

