/*
 * Decompiled with CFR 0.152.
 */
package net.sf.picard.illumina.parser;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.sf.picard.PicardException;
import net.sf.picard.illumina.BarcodeUtil;
import net.sf.picard.illumina.parser.BarcodeParser;
import net.sf.picard.illumina.parser.CifParser;
import net.sf.picard.illumina.parser.CnfParser;
import net.sf.picard.illumina.parser.EndType;
import net.sf.picard.illumina.parser.IlluminaDataProvider;
import net.sf.picard.illumina.parser.IlluminaDataType;
import net.sf.picard.illumina.parser.IlluminaFileUtil;
import net.sf.picard.illumina.parser.IlluminaParser;
import net.sf.picard.illumina.parser.IntParser;
import net.sf.picard.illumina.parser.NseParser;
import net.sf.picard.illumina.parser.PrbParser;
import net.sf.picard.illumina.parser.QhgParser;
import net.sf.picard.illumina.parser.QseqParser;
import net.sf.picard.illumina.parser.ReadConfiguration;
import net.sf.picard.illumina.parser.SeqParser;
import net.sf.picard.illumina.parser.Sig2Parser;
import net.sf.picard.illumina.parser.TiledIlluminaFile;
import net.sf.picard.util.Log;
import net.sf.samtools.util.AsciiLineReader;

public class IlluminaDataProviderFactory {
    private static final Log log = Log.getInstance(IlluminaDataProviderFactory.class);
    private final File basecallDirectory;
    private final int lane;
    private final Integer barcodeCycle;
    private final Integer barcodeLength;
    private final Set<IlluminaDataType> dataTypes = EnumSet.noneOf(IlluminaDataType.class);
    private File rawIntensityDirectory;
    private Boolean pairedEnd;
    private BaseCallerVersion baseCallerVersion;
    private ImageAnalyzerVersion imageAnalyzerVersion;
    private boolean prepared = false;
    private boolean allowZeroLengthFirstEnd = false;
    private final ReadConfiguration readConfiguration = new ReadConfiguration();

    public IlluminaDataProviderFactory(File basecallDirectory, int lane, IlluminaDataType ... dataTypes) {
        this.basecallDirectory = basecallDirectory;
        this.lane = lane;
        this.barcodeCycle = null;
        this.barcodeLength = null;
        this.dataTypes.addAll(Arrays.asList(dataTypes));
    }

    public IlluminaDataProviderFactory(File basecallDirectory, int lane, int barcodeCycle, int barcodeLength, IlluminaDataType ... dataTypes) {
        this.basecallDirectory = basecallDirectory;
        this.lane = lane;
        if (barcodeCycle < 1) {
            throw new IllegalArgumentException("barcodeCycle is < 1: " + barcodeCycle);
        }
        if (barcodeLength < 1) {
            throw new IllegalArgumentException("barcodeLength is < 1: " + barcodeLength);
        }
        this.barcodeCycle = barcodeCycle;
        this.barcodeLength = barcodeLength;
        this.dataTypes.addAll(Arrays.asList(dataTypes));
    }

    public List<Integer> getTiles() {
        if (!this.prepared) {
            this.computeReadConfiguration();
        }
        TiledIlluminaFile[] tiledFiles = this.baseCallerVersion == BaseCallerVersion.Bustard_1_1 ? IlluminaFileUtil.getNonEndedIlluminaBasecallFiles(this.basecallDirectory, "seq", this.lane) : IlluminaFileUtil.getEndedIlluminaBasecallFiles(this.basecallDirectory, "qseq", this.lane, 1);
        ArrayList<Integer> ret = new ArrayList<Integer>(tiledFiles.length);
        for (TiledIlluminaFile tiledFile : tiledFiles) {
            ret.add(tiledFile.tile);
        }
        return ret;
    }

    public IlluminaDataProvider makeDataProvider() {
        return this.makeDataProvider(null);
    }

    public IlluminaDataProvider makeDataProvider(List<Integer> tiles) {
        if (this.dataTypes.isEmpty()) {
            throw new PicardException("No data types have been specified for basecall output " + this.basecallDirectory + ", lane " + this.lane);
        }
        if (!this.prepared) {
            this.computeReadConfiguration();
        }
        boolean madeQseqParser = false;
        ArrayList<IlluminaParser> parsers = new ArrayList<IlluminaParser>();
        for (IlluminaDataType dataType : this.dataTypes) {
            switch (dataType) {
                case Barcodes: {
                    parsers.add(new BarcodeParser(this.readConfiguration, this.basecallDirectory, this.lane, tiles));
                    break;
                }
                case BaseCalls: {
                    if (this.baseCallerVersion == BaseCallerVersion.Bustard_1_1) {
                        parsers.add(new SeqParser(this.readConfiguration, this.basecallDirectory, this.lane, tiles));
                        break;
                    }
                    if (madeQseqParser) break;
                    this.makeQseqParsers(parsers, tiles);
                    madeQseqParser = true;
                    break;
                }
                case Noise: {
                    if (this.imageAnalyzerVersion == ImageAnalyzerVersion.rta) {
                        parsers.add(new CnfParser(this.readConfiguration, this.rawIntensityDirectory, this.lane, tiles));
                        break;
                    }
                    parsers.add(new NseParser(this.readConfiguration, this.rawIntensityDirectory, this.lane, tiles));
                    break;
                }
                case PF: {
                    if (this.baseCallerVersion == BaseCallerVersion.Bustard_1_1) {
                        parsers.add(new QhgParser(this.readConfiguration, this.basecallDirectory, this.lane, tiles));
                        break;
                    }
                    if (madeQseqParser) break;
                    this.makeQseqParsers(parsers, tiles);
                    madeQseqParser = true;
                    break;
                }
                case ProcessedIntensities: {
                    parsers.add(new Sig2Parser(this.readConfiguration, this.basecallDirectory, this.lane, tiles));
                    break;
                }
                case QualityScores: {
                    if (this.baseCallerVersion == BaseCallerVersion.Bustard_1_1) {
                        parsers.add(new PrbParser(this.readConfiguration, this.basecallDirectory, this.lane, tiles));
                        break;
                    }
                    if (madeQseqParser) break;
                    this.makeQseqParsers(parsers, tiles);
                    madeQseqParser = true;
                    break;
                }
                case RawIntensities: {
                    if (this.imageAnalyzerVersion == ImageAnalyzerVersion.rta) {
                        parsers.add(new CifParser(this.readConfiguration, this.rawIntensityDirectory, this.lane, tiles));
                        break;
                    }
                    parsers.add(new IntParser(this.readConfiguration, this.rawIntensityDirectory, this.lane, tiles));
                }
            }
        }
        return new IlluminaDataProvider(this.readConfiguration.isBarcoded(), this.readConfiguration.isPairedEnd(), parsers, this.basecallDirectory, this.lane);
    }

    private void makeQseqParsers(List<IlluminaParser> parsers, List<Integer> tiles) {
        if (this.isBarcodeAwareBaseCaller()) {
            int firstEndFileNumber = 1;
            int secondEndFileNumber = 2;
            int barcodeFileNumber = -1;
            if (this.readConfiguration.isBarcoded()) {
                if (this.readConfiguration.getBarcodeStart() < this.readConfiguration.getFirstStart()) {
                    barcodeFileNumber = 1;
                    firstEndFileNumber = 2;
                    secondEndFileNumber = 3;
                } else if (!this.readConfiguration.isPairedEnd()) {
                    barcodeFileNumber = 2;
                } else if (this.readConfiguration.getBarcodeStart() < this.readConfiguration.getSecondStart()) {
                    barcodeFileNumber = 2;
                    secondEndFileNumber = 3;
                } else {
                    barcodeFileNumber = 3;
                }
            }
            parsers.add(new QseqParser(this.readConfiguration, this.basecallDirectory, this.lane, EndType.FIRST, firstEndFileNumber, tiles));
            if (this.readConfiguration.isPairedEnd()) {
                parsers.add(new QseqParser(this.readConfiguration, this.basecallDirectory, this.lane, EndType.SECOND, secondEndFileNumber, tiles));
            }
            if (this.readConfiguration.isBarcoded()) {
                parsers.add(new QseqParser(this.readConfiguration, this.basecallDirectory, this.lane, EndType.BARCODE, barcodeFileNumber, tiles));
            }
        } else {
            parsers.add(new QseqParser(this.readConfiguration, this.basecallDirectory, this.lane, EndType.FIRST, tiles));
            if (this.pairedEnd.booleanValue() || this.readConfiguration.isBarcoded() && this.readConfiguration.getBarcodeRead() == EndType.SECOND) {
                parsers.add(new QseqParser(this.readConfiguration, this.basecallDirectory, this.lane, EndType.SECOND, tiles));
            }
        }
    }

    public boolean intensitiesAvailable(int tile) {
        if (!this.prepared) {
            this.computeReadConfiguration();
        }
        if (this.imageAnalyzerVersion == ImageAnalyzerVersion.rta) {
            return CifParser.cifExists(this.rawIntensityDirectory, this.lane, tile);
        }
        return IntParser.intExists(this.rawIntensityDirectory, this.lane, tile);
    }

    public void computeReadConfiguration() {
        if (this.prepared) {
            throw new IllegalStateException("Already prepared");
        }
        if (this.baseCallerVersion == null || this.imageAnalyzerVersion == null) {
            this.detectPipelineVersion();
        }
        if (this.rawIntensityDirectory == null) {
            switch (this.imageAnalyzerVersion) {
                case Firecrest_1_1: 
                case Firecrest_1_3: 
                case Firecrest_1_4: 
                case rta: {
                    this.rawIntensityDirectory = this.basecallDirectory.getParentFile();
                    break;
                }
                default: {
                    throw new IllegalStateException("Could not determine image analyzer version for " + this.basecallDirectory + "; lane " + this.lane);
                }
            }
        }
        boolean qseqIsBarcodeAware = false;
        switch (this.baseCallerVersion) {
            case rta: 
            case Bustard_1_3: 
            case Bustard_1_4: {
                this.computeReadConfigurationFromBarcodeUnawareQseq();
                break;
            }
            case Bustard_1_5: 
            case Bustard_1_6: {
                this.computeReadConfigurationFromBarcodeAwareQseq();
                qseqIsBarcodeAware = true;
                break;
            }
            case Bustard_1_1: {
                this.computeReadConfiguationFrom_1_1();
                break;
            }
            default: {
                throw new IllegalStateException("Could not determine base caller version for " + this.basecallDirectory + "; lane " + this.lane);
            }
        }
        if (!qseqIsBarcodeAware) {
            this.updateReadConfigurationForBarcode();
        }
        this.readConfiguration.assertValid(this.allowZeroLengthFirstEnd);
        this.prepared = true;
    }

    private void computeReadConfigurationFromBarcodeUnawareQseq() {
        if (this.pairedEnd == null) {
            this.pairedEnd = IlluminaFileUtil.endedIlluminaBasecallFilesExist(this.basecallDirectory, "qseq", this.lane, 2);
        }
        this.readConfiguration.setPairedEnd(this.pairedEnd);
        if (this.readConfiguration.getFirstLength() == 0) {
            this.readConfiguration.setFirstStart(1);
            File firstEnd = IlluminaFileUtil.getEndedIlluminaBasecallFiles((File)this.basecallDirectory, (String)"qseq", (int)this.lane, (int)1)[0].file;
            this.readConfiguration.setFirstEnd(QseqParser.getReadLength(firstEnd));
        }
        if (this.pairedEnd.booleanValue() && this.readConfiguration.getSecondLength() == 0) {
            File secondEnd = IlluminaFileUtil.getEndedIlluminaBasecallFiles((File)this.basecallDirectory, (String)"qseq", (int)this.lane, (int)2)[0].file;
            int end2Length = QseqParser.getReadLength(secondEnd);
            this.readConfiguration.setSecondStart(this.readConfiguration.getFirstLength() + 1);
            this.readConfiguration.setSecondEnd(this.readConfiguration.getFirstLength() + end2Length);
        }
    }

    private void computeReadConfigurationFromBarcodeAwareQseq() {
        int numQseqs;
        File[] qseqs = new File[3];
        qseqs[0] = IlluminaFileUtil.getEndedIlluminaBasecallFiles((File)this.basecallDirectory, (String)"qseq", (int)this.lane, (int)1)[0].file;
        for (int end = 2; end <= 3; ++end) {
            TiledIlluminaFile[] files = IlluminaFileUtil.getEndedIlluminaBasecallFiles(this.basecallDirectory, "qseq", this.lane, end);
            qseqs[end - 1] = files.length > 0 ? files[0].file : null;
        }
        this.readConfiguration.setBarcoded(this.barcodeCycle != null);
        int n = numQseqs = this.readConfiguration.isBarcoded() ? 2 : 1;
        if (this.pairedEnd == null) {
            this.pairedEnd = qseqs[numQseqs] != null;
        }
        if (this.pairedEnd.booleanValue()) {
            ++numQseqs;
        }
        this.readConfiguration.setPairedEnd(this.pairedEnd);
        int[] qseqLengths = new int[numQseqs];
        for (int i = 0; i < numQseqs; ++i) {
            qseqLengths[i] = QseqParser.getReadLength(qseqs[i]);
        }
        int firstEndIndex = 0;
        int secondEndIndex = 1;
        int barcodeIndex = -1;
        if (this.readConfiguration.isBarcoded()) {
            int cyclesSoFar = 1;
            for (barcodeIndex = 0; this.barcodeCycle > cyclesSoFar && barcodeIndex < qseqLengths.length; cyclesSoFar += qseqLengths[barcodeIndex], ++barcodeIndex) {
            }
            if (this.barcodeCycle != cyclesSoFar) {
                throw new PicardException("Barcode cycle " + this.barcodeCycle + " does not fall on qseq boundary for barcode-aware qseqs.");
            }
            if (barcodeIndex <= secondEndIndex) {
                ++secondEndIndex;
            }
            if (barcodeIndex == firstEndIndex) {
                ++firstEndIndex;
            }
        }
        ReadConfiguration.InclusiveRange[] ranges = new ReadConfiguration.InclusiveRange[numQseqs];
        ranges[firstEndIndex] = this.readConfiguration.getFirstRange();
        if (this.pairedEnd.booleanValue()) {
            ranges[secondEndIndex] = this.readConfiguration.getSecondRange();
        }
        if (this.readConfiguration.isBarcoded()) {
            ranges[barcodeIndex] = this.readConfiguration.getBarcodeRange();
        }
        int cyclesSoFar = 0;
        for (int i = 0; i < ranges.length; ++i) {
            ranges[i].setStart(++cyclesSoFar);
            ranges[i].setEnd(cyclesSoFar += qseqLengths[i] - 1);
        }
        if (this.readConfiguration.isBarcoded() && this.readConfiguration.getBarcodeLength() != this.barcodeLength.intValue()) {
            throw new PicardException("Barcode length from qseq file (" + this.readConfiguration.getBarcodeLength() + ") != barcode length from command line (" + this.barcodeLength + ").");
        }
    }

    private void computeReadConfiguationFrom_1_1() {
        if (this.rawIntensityDirectory == null) {
            this.rawIntensityDirectory = this.basecallDirectory.getParentFile();
        }
        final Pattern pattern = Pattern.compile("s_" + this.lane + "_([0-9]+)_matrix.txt");
        File[] matrices = new File(this.rawIntensityDirectory, "Matrix").listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File parent, String name) {
                return pattern.matcher(name).matches();
            }
        });
        if (this.pairedEnd == null) {
            if (matrices.length == 1) {
                this.pairedEnd = false;
            } else if (matrices.length == 2) {
                this.pairedEnd = true;
            } else {
                throw new IllegalStateException("Cannot determine if Bustard 1.1 output is PE or not.");
            }
        }
        int firstReadLength = -1;
        if (this.pairedEnd.booleanValue()) {
            Matcher matcher = pattern.matcher(matrices[0].getName());
            if (!matcher.matches()) {
                throw new PicardException("That's unpossible");
            }
            int c1 = Integer.parseInt(matcher.group(1));
            matcher = pattern.matcher(matrices[1].getName());
            if (!matcher.matches()) {
                throw new PicardException("That's unpossible");
            }
            int c2 = Integer.parseInt(matcher.group(1));
            firstReadLength = Math.max(c1, c2) - Math.min(c1, c2);
        }
        this.readConfiguration.setPairedEnd(this.pairedEnd);
        File seqFile = IlluminaFileUtil.getNonEndedIlluminaBasecallFiles((File)this.basecallDirectory, (String)"seq", (int)this.lane)[0].file;
        int totalReadLength = SeqParser.getReadLength(seqFile);
        this.readConfiguration.setFirstStart(1);
        if (this.pairedEnd.booleanValue()) {
            this.readConfiguration.setFirstEnd(firstReadLength);
            this.readConfiguration.setSecondStart(firstReadLength + 1);
            this.readConfiguration.setSecondEnd(totalReadLength);
        } else {
            this.readConfiguration.setFirstEnd(totalReadLength);
        }
    }

    private void updateReadConfigurationForBarcode() {
        if (this.barcodeCycle == null) {
            this.readConfiguration.setBarcoded(false);
            return;
        }
        BarcodeUtil.BarcodePosition barcodePosition = BarcodeUtil.findBarcodeEndAndStart(this.pairedEnd, this.readConfiguration.getFirstLength(), this.readConfiguration.getSecondLength(), this.barcodeCycle, this.barcodeLength);
        this.readConfiguration.setBarcoded(true);
        this.readConfiguration.setBarcodeStart(this.barcodeCycle);
        this.readConfiguration.setBarcodeEnd(this.barcodeCycle + this.barcodeLength - 1);
        this.readConfiguration.setBarcodeRead(barcodePosition.barcodeIsInSecondRead ? EndType.SECOND : EndType.FIRST);
        if (barcodePosition.barcodeIsInSecondRead) {
            if (this.barcodeCycle.intValue() == this.readConfiguration.getSecondStart()) {
                this.readConfiguration.setSecondStart(this.barcodeCycle + this.barcodeLength);
            } else {
                this.readConfiguration.setSecondEnd(this.barcodeCycle - 1);
            }
            if (this.readConfiguration.getSecondLength() == 0) {
                this.readConfiguration.setPairedEnd(false);
                this.setPairedEnd(false);
            }
        } else if (this.barcodeCycle.intValue() == this.readConfiguration.getFirstStart()) {
            this.readConfiguration.setFirstStart(this.barcodeCycle + this.barcodeLength);
        } else {
            this.readConfiguration.setFirstEnd(this.barcodeCycle - 1);
        }
    }

    public void addDataType(IlluminaDataType dataType) {
        this.dataTypes.add(dataType);
    }

    public void removeDataType(IlluminaDataType dataType) {
        this.dataTypes.remove((Object)dataType);
    }

    public void setPairedEnd(boolean pairedEnd) {
        this.pairedEnd = pairedEnd;
    }

    public Boolean isPairedEnd() {
        return this.pairedEnd;
    }

    public void setBaseCallerVersion(BaseCallerVersion baseCallerVersion) {
        this.baseCallerVersion = baseCallerVersion;
    }

    public BaseCallerVersion getBaseCallerVersion() {
        return this.baseCallerVersion;
    }

    public ImageAnalyzerVersion getImageAnalyzerVersion() {
        return this.imageAnalyzerVersion;
    }

    public void setImageAnalyzerVersion(ImageAnalyzerVersion imageAnalyzerVersion) {
        this.imageAnalyzerVersion = imageAnalyzerVersion;
    }

    public void setRawIntensityDirectory(File rawIntensityDirectory) {
        this.rawIntensityDirectory = rawIntensityDirectory;
    }

    private boolean isBarcodeAwareBaseCaller() {
        return this.baseCallerVersion == BaseCallerVersion.Bustard_1_6 || this.baseCallerVersion == BaseCallerVersion.Bustard_1_5;
    }

    public void setAllowZeroLengthFirstEnd(boolean allowZeroLengthFirstEnd) {
        this.allowZeroLengthFirstEnd = allowZeroLengthFirstEnd;
    }

    private <T> T elvisOperator(T v1, T v2) {
        if (v1 != null) {
            return v1;
        }
        return v2;
    }

    private void detectPipelineVersion() {
        File solexaBuildVersion = new File(this.basecallDirectory, ".solexaBuildVersion");
        if (solexaBuildVersion.exists()) {
            AsciiLineReader reader;
            try {
                reader = new AsciiLineReader((InputStream)new FileInputStream(solexaBuildVersion));
            }
            catch (FileNotFoundException e) {
                throw new PicardException("Unexpected FileNotFound: " + solexaBuildVersion, e);
            }
            String version = reader.readLine();
            log.info("solexaBuildVersion: ", version);
            reader.close();
            if (version.startsWith("1.6")) {
                this.baseCallerVersion = this.elvisOperator(this.baseCallerVersion, BaseCallerVersion.Bustard_1_6);
                this.imageAnalyzerVersion = this.elvisOperator(this.imageAnalyzerVersion, ImageAnalyzerVersion.rta);
            } else if (version.startsWith("1.5")) {
                this.baseCallerVersion = this.elvisOperator(this.baseCallerVersion, BaseCallerVersion.Bustard_1_5);
                this.imageAnalyzerVersion = this.elvisOperator(this.imageAnalyzerVersion, ImageAnalyzerVersion.rta);
            } else if (version.startsWith("1.4")) {
                this.baseCallerVersion = this.elvisOperator(this.baseCallerVersion, BaseCallerVersion.Bustard_1_4);
                this.imageAnalyzerVersion = this.elvisOperator(this.imageAnalyzerVersion, ImageAnalyzerVersion.rta);
            } else if (version.startsWith("1.3")) {
                this.baseCallerVersion = this.elvisOperator(this.baseCallerVersion, BaseCallerVersion.Bustard_1_3);
                this.imageAnalyzerVersion = this.elvisOperator(this.imageAnalyzerVersion, ImageAnalyzerVersion.Firecrest_1_3);
            } else if (version.startsWith("1.1")) {
                this.baseCallerVersion = this.elvisOperator(this.baseCallerVersion, BaseCallerVersion.Bustard_1_1);
                this.imageAnalyzerVersion = this.elvisOperator(this.imageAnalyzerVersion, ImageAnalyzerVersion.Firecrest_1_1);
            } else {
                log.info("Unrecognized solexaBuildVersion, using fallback to detect version. ", version);
            }
        } else {
            log.info(solexaBuildVersion, " not found, using fallback to detect version.");
        }
        if (new File(this.basecallDirectory.getParentFile(), "BaseCalls").exists()) {
            log.info("Found BaseCalls directory.");
            if (IlluminaFileUtil.endedIlluminaBasecallFilesExist(this.basecallDirectory, "qseq", this.lane, 3)) {
                log.info("Found barcode qseq file.");
                this.baseCallerVersion = this.elvisOperator(this.baseCallerVersion, BaseCallerVersion.Bustard_1_5);
            } else {
                log.info("Did not find barcode qseq file.");
                this.baseCallerVersion = this.elvisOperator(this.baseCallerVersion, BaseCallerVersion.Bustard_1_4);
            }
            this.imageAnalyzerVersion = this.elvisOperator(this.imageAnalyzerVersion, ImageAnalyzerVersion.rta);
        } else if (IlluminaFileUtil.endedIlluminaBasecallFilesExist(this.basecallDirectory, "qseq", this.lane, 1)) {
            log.info("Found end-specific qseq file.");
            this.baseCallerVersion = this.elvisOperator(this.baseCallerVersion, BaseCallerVersion.Bustard_1_3);
            this.imageAnalyzerVersion = this.elvisOperator(this.imageAnalyzerVersion, ImageAnalyzerVersion.Firecrest_1_3);
        } else if (IlluminaFileUtil.nonEndedIlluminaBasecallFilesExist(this.basecallDirectory, "seq", this.lane)) {
            log.info("Found non-end-specific qseq file.");
            this.baseCallerVersion = this.elvisOperator(this.baseCallerVersion, BaseCallerVersion.Bustard_1_1);
            this.imageAnalyzerVersion = this.elvisOperator(this.imageAnalyzerVersion, ImageAnalyzerVersion.Firecrest_1_1);
        } else {
            throw new IllegalStateException("Cannot determine pipeline version for basecall directory: " + this.basecallDirectory);
        }
        log.info("BaseCallerVersion: " + (Object)((Object)this.baseCallerVersion));
        log.info("ImageAnalyzerVersion: " + (Object)((Object)this.imageAnalyzerVersion));
    }

    public static enum ImageAnalyzerVersion {
        Firecrest_1_1,
        Firecrest_1_3,
        Firecrest_1_4,
        rta;

    }

    public static enum BaseCallerVersion {
        Bustard_1_1,
        Bustard_1_3,
        Bustard_1_4,
        rta,
        Bustard_1_5,
        Bustard_1_6;

    }
}

